
/* pngexif - to extract or insert EXIF data from/into PNG files */

/* (c) Willem van Schaik, July 2017 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pngexif.h"

#define EXIF "eXIf"
#define IHDR "IHDR"
#define IEND "IEND"

static char prog_name[256];
static char file_in[256];
static char file_out[256];
static char file_exif[256];

int main (int, char*[]);
void exif_extract (FILE*, FILE*);
void exif_insert (FILE*, FILE*, FILE*);
unsigned char inout (FILE*, FILE*);
unsigned char inonly (FILE*);
unsigned char outonly (FILE*, unsigned char);
unsigned int crc32 (unsigned int, unsigned char*, int);


int 
main (int argc, char * argv[]) {
  FILE *fp_in;
  FILE *fp_out;
  FILE *fp_exif;
  int val;
  unsigned char c;
  unsigned char b;
  char chnk [5];
  int l;
  int i;

  if (argc >= 1)
  {
    strcpy (prog_name, argv[0]);
  }

  if (argc == 1)
  {
    /* PNG input is stdin, extract EXIF and write to stdout */

    exif_extract (stdin, stdout);

  }
  else if (argc == 2)
  {
    if ((strncmp(argv[1], "-h", 2) == 0) ||
        (strncmp(argv[1], "-?", 2) == 0))
    {
      fprintf (stderr, "Usage: \n");
      fprintf (stderr, "%s                          - read PNG from stdin and write EXIF to stdout) \n", prog_name);
      fprintf (stderr, "%s <png-infile> <exif-file> - read PNG from infile and write EXIF to outfile) \n", prog_name);
      fprintf (stderr, "%s -e[xif] <exif-file>      - read PNG from stdin, add EXIF chunk and write PNG to stdout) \n", prog_name);
      fprintf (stderr, "%s -e[xif] <exif-file> <png-infile> <png-outfile> \n", prog_name);
      l = strlen(prog_name);
      for (i = 0; i < l; i++)
      {
        fprintf (stderr, " ");
      }
      fprintf (stderr,   "                          - read PNG from infile, add EXIF chunk and write to outfile) \n");
      fprintf (stderr, "\n");

      exit (1);
    }
  }
  else if (argc == 3)
  {
    if (strncmp(argv[1], "-e", 2) == 0)
    {
      strcpy (file_exif, argv[2]);

      /* PNG input is stdin, add EXIF chunk and write PNG to stdout */

      if ((fp_exif = fopen (file_exif, "rb")) == NULL)
      {
        fprintf (stderr, "%s: %s not found\n", prog_name, file_exif);
        exit (1);
      }

      exif_insert (stdin, stdout, fp_exif);

      fclose (fp_exif);
    }

    else
    {
      strcpy (file_in, argv[1]);
      strcpy (file_exif, argv[2]);

      /* command-line has PNG input and EXIF output filenames */

      if ((fp_in = fopen (file_in, "rb")) == NULL)
      {
        fprintf (stderr, "%s: %s not found\n", prog_name, file_in);
        exit (1);
      }

      if ((fp_out = fopen (file_exif, "wb")) == NULL)
      {
        fprintf (stderr, "%s: %s can't be created\n", prog_name, file_exif);
        exit (1);
      }

      exif_extract (fp_in, fp_out);

      fclose (fp_in);
      fclose (fp_out);
    }
  }
  else if (argc == 5)
  {
    if (strncmp(argv[1], "-e", 2) == 0)
    {
      strcpy (file_exif, argv[2]);
      strcpy (file_in, argv[3]);
      strcpy (file_out, argv[4]);

      /* PNG input is filein, add EXIF chunk and write PNG to fileout */

      if ((fp_in = fopen (file_in, "rb")) == NULL)
      {
        fprintf (stderr, "%s: %s not found\n", prog_name, file_in);
        exit (1);
      }

      if ((fp_out = fopen (file_out, "wb")) == NULL)
      {
        fprintf (stderr, "%s: %s can't be created\n", prog_name, file_out);
        exit (1);
      }

      if ((fp_exif = fopen (file_exif, "rb")) == NULL)
      {
        fprintf (stderr, "%s: %s not found\n", prog_name, file_exif);
        exit (1);
      }

      exif_insert (fp_in, fp_out, fp_exif);

      fclose (fp_in);
      fclose (fp_out);
      fclose (fp_exif);
    }
  }

  return 0;
} // end of main


void 
exif_extract (FILE *fp_in, FILE *fp_out)
{
  char chnk [5];
  unsigned char c;
  int l;
  int i;

  // read PNG header
  for (i = 0; i < 8; i++)
  {
    c = inonly(fp_in);
    if (c != png_sig[i])
    {
      fprintf (stderr, "pngexif: PNG signature error (byte %d)\n", i);
      exit (1);
    }
  }

  // process chunks
  strcpy (chnk, "");
  do
  {
    // get chunk size
    l = 0;
    for (i = 0; i < 4; i++)
    {
      c = inonly (fp_in);
      l = (l << 8) + c;
    }

    // get chunk name
    strcpy (chnk, "");
    for (i = 0; i < 4; i++)
    {
      c = inonly (fp_in);
      chnk[i] = (char) c;
    }
    chnk[4] = '\0';

    fprintf (stderr, "%s (%3d )\n", chnk, l);

    // check chunk name for exif and extract
    if (strcmp(chnk, EXIF) == 0)
    {
      for (i = 0; i < l; i++)
      {
        c = inout (fp_in, fp_out);
      }
    }
    else
    {
      // read chunk data
      for (i = 0; i < l; i++)
      {
        c = inonly (fp_in);
      }
    }

    // read checksum
    for (i = 0; i < 4; i++)
    {
      c = inonly (fp_in);
    }

  }
  while (strcmp (chnk, IEND) != 0);

  return;
} // end of exif_extract


void 
exif_insert (FILE *fp_in, FILE *fp_out, FILE *fp_exif)
{
  int val;
  unsigned char c;
  unsigned char b;
  unsigned int csum = 0;
  unsigned int crc = 0;
  char chnk [5];
  int l, ll;
  int i;
  int ihdr_done = 0;

  // copy PNG header
  for (i = 0; i < 8; i++)
  {
    if ((val = fgetc(fp_in)) == EOF)
    {
      fprintf (stderr, "pngexif: PNG file too short\n");
      exit (1);
    }
    else
    {
      fputc (val, fp_out);

      if (val != (int) png_sig[i])
      {
        fprintf (stderr, "pngexif: PNG signature error (byte %d)\n", i);
        exit (1);
      }
    }
  }

  // process chunks
  strcpy (chnk, "");
  do
  {
    if (ihdr_done == 1)
    {
      // determine size of EXIF file
      l = 0;
      while ((val = fgetc(fp_exif)) != EOF)
      {
        l++;
      }
      rewind (fp_exif);

      // write EXIF chunk size
      ll = l;
      for (i = 0; i < 4; i++)
      {
        c = ((ll & 0xFF000000) >> 24) & 0xFF;
        ll = (ll << 8);
        outonly (fp_out, c);
      }

      // write EXIF chunk name
      crc = 0;
      strcpy (chnk, EXIF);
      for (i = 0; i < 4; i++)
      {
        c = (unsigned char) chnk[i];
        outonly (fp_out, c);
        crc = crc32 (crc, &c, 1);
      }

      // write EXIF data
      for (i = 0; i < l; i++)
      {
        c = inout (fp_exif, fp_out);
        crc = crc32 (crc, &c, 1);
      }

      // write EXIF checksum
      for (i = 0; i < 4; i++)
      {
        c = (unsigned char) ((crc >> 8 * (3 - i)) & 0xFF);
        outonly (fp_out, c);
      }

      ihdr_done = 0;
    }

    // get chunk size
    l = 0;
    for (i = 0; i < 4; i++)
    {
      c = inout (fp_in, fp_out);
      l = (l << 8) + c;
    }

    // get chunk name
    crc = 0;
    strcpy (chnk, "");
    for (i = 0; i < 4; i++)
    {
      c = inout (fp_in, fp_out);
      crc = crc32 (crc, &c, 1);
      chnk[i] = (char) c;
    }
    chnk[4] = '\0';

    fprintf (stderr, "%s (%3d )\n", chnk, l);

    if (strcmp(chnk, IHDR) == 0)
    {
      ihdr_done = 1;
    }

    // copy chunk data
    for (i = 0; i < l; i++)
    {
      c = inout (fp_in, fp_out);
      crc = crc32 (crc, &c, 1);
    }

    // copy checksum
    csum = 0;
    for (i = 0; i < 4; i++)
    {
      c = inonly (fp_in);
      csum = (csum << 8) + (int) c;

      b = (unsigned char) ((crc >> 8 * (3 - i)) & 0xFF);
      outonly (fp_out, b);
    }
  }
  while (strcmp (chnk, "IEND") != 0);

  return;
} // end of exif_insert


unsigned char
inout (FILE *fp_in, FILE *fp_out)
{
  int val, out;

  if ((val = fgetc(fp_in)) == EOF)
  {
    fprintf (stderr, "%s: %s too short\n", prog_name, file_in);
    exit (1);
  }
  else
  {
    if ((out = fputc(val, fp_out)) == EOF)
    {
      fprintf (stderr, "%s: %s couln't be written\n", prog_name, file_out);
      exit (1);
    }
  }

  return ((unsigned char) val);
} // end of inout()


unsigned char
inonly (FILE *fp_in)
{
  int val;

  if ((val = fgetc(fp_in)) == EOF)
  {
    fprintf (stderr, "%s: %s too short\n", prog_name, file_in);
    exit (1);
  }

  return ((unsigned char) val);
} // end of inonly()


unsigned char
outonly (FILE *fp_out, unsigned char c)
{
  int out;
  int val;

  val = (int) c;
  if ((out = fputc(val, fp_out)) == EOF)
  {
    fprintf (stderr, "%s: %s couln't be written\n", prog_name, file_out);
    exit (1);
  }

  return ((unsigned char) val);
} // end of outonly()


unsigned int
crc32 (unsigned int crc, unsigned char *buf, int len)
{
  unsigned char *end;

  crc = ~crc;
  for (end = buf + len; buf < end; ++buf)
  {
    crc = crc32_table[(crc ^ *buf) & 0xff] ^ (crc >> 8);
  }

  return ~crc;
} // end of crc32()


// end of code
